<?php
require_once '../../core/init.php';
require_once 'sidebar.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../auth/admin/login.php");
    exit;
}

$agentId = $_GET['id'] ?? null;
if (!$agentId) {
    echo "<p class='text-red-600'>No merchant ID provided.</p>";
    exit;
}

$merchant = $conn->query("SELECT * FROM submerchants WHERE id = " . intval($agentId))->fetch_assoc();
if (!$merchant) {
    echo "<p class='text-red-600'>Merchant not found.</p>";
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_changes'])) {
    $updatedBy = $_SESSION['user_id'];
    $merchantId = intval($agentId);
    $updatedFields = [];

    foreach ($_POST as $field => $newValue) {
        if (!array_key_exists($field, $merchant) || $field === 'save_changes') continue;
        $oldValue = $merchant[$field];
        if ($newValue != $oldValue) {
            $stmt = $conn->prepare("UPDATE submerchants SET `$field` = ?, last_updated_by = ?, last_updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("sii", $newValue, $updatedBy, $merchantId);
            $stmt->execute();

            $auditStmt = $conn->prepare("INSERT INTO merchant_audit_log (merchant_id, field_changed, old_value, new_value, updated_by, updated_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $auditStmt->bind_param("isssi", $merchantId, $field, $oldValue, $newValue, $updatedBy);
            $auditStmt->execute();

            $updatedFields[] = $field;
        }
    }

    if (!empty($updatedFields)) {
        header("Location: merchant.php?id=" . $merchantId . "&updated=1");
        exit;
    }
    $merchant = $conn->query("SELECT * FROM submerchants WHERE id = " . intval($agentId))->fetch_assoc(); // refresh data
}

$api = new CycleAPI();
$transactions = $api->getClientTransactions(date('Y-m-d', strtotime('-30 days')), date('Y-m-d'));
$filteredTransactions = array_filter($transactions, function ($tx) use ($merchant) {
    return isset($tx['PosID']) && $tx['PosID'] == $merchant['cycle_agent_id'];
});
$totalVolume = array_reduce($filteredTransactions, fn($carry, $tx) => $carry + ($tx['Amount'] ?? 0), 0);
$totalTx = count($filteredTransactions);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Merchant Profile | Swirl</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
    function switchTab(tabId, el) {
      document.querySelectorAll('.tab-content').forEach(div => div.classList.add('hidden'));
      document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('border-purple-600', 'text-purple-700', 'font-semibold');
        btn.classList.add('text-gray-600');
      });
      document.getElementById(tabId).classList.remove('hidden');
      el.classList.add('border-purple-600', 'text-purple-700', 'font-semibold');
      el.classList.remove('text-gray-600');
    }

    function toggleEditMode() {
      document.querySelectorAll('.editable').forEach(el => {
        const field = el.dataset.field;
        const val = el.innerText;
        el.innerHTML = `<input type="text" name="${field}" value="${val}" class="w-full border rounded p-1 text-sm" />`;
      });
      document.getElementById('editBtn').classList.add('hidden');
      document.getElementById('saveBtn').classList.remove('hidden');
    }

    document.addEventListener('DOMContentLoaded', () => {
      const savedTab = localStorage.getItem('activeTabId');
      const savedButton = [...document.querySelectorAll('.tab-button')].find(btn => btn.dataset.tab === savedTab);

      if (savedButton) {
        switchTab(savedTab, savedButton);
      } else {
        const firstTab = document.querySelector('.tab-button');
        if (firstTab) switchTab(firstTab.getAttribute('data-tab'), firstTab);
      }

      document.querySelectorAll('.tab-button').forEach(btn => {
        btn.addEventListener('click', () => {
          localStorage.setItem('activeTabId', btn.dataset.tab);
        });
      });
    });
  </script>
</head>
<body class="bg-gray-50 text-gray-800">
  <div class="flex min-h-screen">
    
    <!-- Sidebar -->
    <?php require_once 'sidebar.php'; ?>

    <main class="ml-64 flex-1 p-8">
      <div class="mb-6">
        <a href="merchants.php" class="text-purple-600 hover:underline text-sm">← Back to Merchants</a>
        <h1 class="text-2xl font-bold mt-2">Merchant: <?= htmlspecialchars($merchant['name']) ?></h1>
        <p class="text-gray-500">Transaction Volume (30 days): <strong>R<?= number_format($totalVolume, 2) ?></strong> | Total Transactions: <?= $totalTx ?></p>
        <?php if (isset($_GET['updated'])): ?>
          <p class="text-green-600 text-sm mt-2">Merchant profile updated successfully.</p>
        <?php endif; ?>
      </div>

      <form method="POST">
        <input type="hidden" name="merchant_id" value="<?= $merchant['id'] ?>" />

        <div class="mb-4 flex gap-4">
          <button type="button" id="editBtn" onclick="toggleEditMode()" class="bg-purple-600 text-white px-4 py-2 rounded">Edit</button>
          <button type="submit" name="save_changes" id="saveBtn" class="bg-green-600 text-white px-4 py-2 rounded hidden">Save Changes</button>
        </div>

        <!-- Tabs -->
        <div class="mb-6">
          <div class="flex gap-6 border-b">
            <button data-tab="tab-overview" onclick="switchTab('tab-overview', this)" class="tab-button pb-2 border-b-2 border-purple-600 text-purple-700 font-semibold">Overview</button>
            <button data-tab="tab-compliance" onclick="switchTab('tab-compliance', this)" class="tab-button pb-2 text-gray-600 hover:text-purple-600">Compliance</button>
            <button data-tab="tab-monitoring" onclick="switchTab('tab-monitoring', this)" class="tab-button pb-2 text-gray-600 hover:text-purple-600">Monitoring</button>
            <button data-tab="tab-termination" onclick="switchTab('tab-termination', this)" class="tab-button pb-2 text-gray-600 hover:text-purple-600">Termination</button>
          </div>
        </div>

        <?php
        function renderCards($merchant, $filterFn) {
          echo '<div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6">';
          foreach ($merchant as $key => $value) {
            if (call_user_func($filterFn, $key)) {
              echo '<div class="bg-white p-4 rounded shadow">';
              echo '<p class="text-sm text-gray-500 font-semibold">' . ucwords(str_replace('_', ' ', $key)) . '</p>';
              $display = ($value === null || $value === '') ? '-' : htmlspecialchars($value);
              echo '<p class="text-sm text-black mt-1 break-words editable" data-field="' . $key . '">' . $display . '</p>';
              echo '</div>';
            }
          }
          echo '</div>';
        }
        ?>

        <div id="tab-overview" class="tab-content">
          <?php
            renderCards($merchant, fn($key) => !preg_match('/(doc_|check|visit|termination|monitoring|aml|ofac|abac|kyc|due_diligence)/', $key));
          ?>
        </div>

        <div id="tab-compliance" class="tab-content hidden">
          <?php
            renderCards($merchant, fn($key) => preg_match('/(kyc|due_diligence|match|ofac|aml|abac|site_visit)/', $key));
          ?>
        </div>

        <div id="tab-monitoring" class="tab-content hidden">
          <?php
            renderCards($merchant, fn($key) => preg_match('/(monitoring|trading|reviewed)/', $key));
          ?>
        </div>

        <div id="tab-termination" class="tab-content hidden">
          <?php
            renderCards($merchant, fn($key) => str_contains($key, 'termination'));
          ?>
        </div>

      </form>
    </main>
  </div>
</body>
</html>
