<?php
require_once 'config.php';
require_once 'CycleAPI.php';

$api = new CycleAPI();
$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
    die("DB Connection Failed: " . $conn->connect_error);
}

$from = $_GET['from'] ?? date('Y-m-d', strtotime('-3 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$selectedAgent = $_GET['agent'] ?? '';
$export = $_GET['export'] ?? '';

// Load agent list
$agents = file_exists("agents.json") ? json_decode(file_get_contents("agents.json"), true) : [];
$agentMap = [];
foreach ($agents as $a) {
    $agentMap[$a['AgentID']] = $a['Name'];
}

$transactions = $api->getClientTransactions($from, $to);

// Filter by agent
if (!empty($selectedAgent)) {
    $transactions = array_filter($transactions, fn($tx) => ($tx['PosID'] ?? '') == $selectedAgent);
}

// Group by agent and date
$grouped = [];
foreach ($transactions as $tx) {
    $posId = $tx['PosID'] ?? 'unknown';
    $agentName = $agentMap[$posId] ?? 'Unknown';
    $date = date('Y-m-d', strtotime($tx['Date']));
    $grouped[$posId][$date][] = $tx;
}

// Export CSV if requested
if ($export) {
    header('Content-Type: text/csv');
    header("Content-Disposition: attachment; filename=settlements_$export.csv");
    $out = fopen("php://output", "w");
    fputcsv($out, ["Agent", "Date", "Total", "Fees", "Net", "Status"]);
    foreach ($grouped as $agentId => $dates) {
        foreach ($dates as $date => $txs) {
            $total = $fees = $net = 0;
            $feePercent = 2.5;
            $stmt = $conn->prepare("SELECT fee_percent FROM submerchants WHERE cycle_agent_id = ? LIMIT 1");
            $stmt->bind_param("i", $agentId);
            $stmt->execute();
            $res = $stmt->get_result();
            if ($r = $res->fetch_assoc()) $feePercent = $r['fee_percent'];

            foreach ($txs as $tx) {
                if (($tx['Substate'] ?? null) == 411) {
                    $amt = $tx['Amount'];
                    $fee = $amt * ($feePercent / 100);
                    $total += $amt;
                    $fees += $fee;
                    $net += $amt - $fee;
                }
            }

            $statusRes = $conn->query("SELECT payout_status FROM settlements WHERE submerchant_id = $agentId AND period_start = '$date' LIMIT 1");
            $status = $statusRes && $statusRes->num_rows ? $statusRes->fetch_assoc()['payout_status'] : 'Unsettled';
            fputcsv($out, [$agentMap[$agentId] ?? 'Unknown', $date, $total, $fees, $net, $status]);
        }
    }
    fclose($out);
    exit;
}

function formatAmount($a) {
    return 'R' . number_format($a, 2);
}
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Settlements</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 text-gray-900 p-6">
    <div class="mb-6 flex justify-between items-center">
        <h1 class="text-2xl font-bold text-purple-700">💰 Settlements</h1>
        <form method="GET" class="flex gap-2 items-center">
            <input type="date" name="from" value="<?= $from ?>" class="p-2 rounded border">
            <input type="date" name="to" value="<?= $to ?>" class="p-2 rounded border">
            <select name="agent" class="p-2 rounded border">
                <option value="">All Agents</option>
                <?php foreach ($agents as $a): ?>
                    <option value="<?= $a['AgentID'] ?>" <?= $selectedAgent == $a['AgentID'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($a['Name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button type="submit" class="bg-purple-600 text-white px-4 py-2 rounded">Filter</button>
            <button name="export" value="all" class="bg-green-600 text-white px-4 py-2 rounded">⬇ Export All CSV</button>
        </form>
    </div>

    <?php foreach ($grouped as $agentId => $dates): ?>
        <?php foreach ($dates as $date => $txs): ?>
            <?php
                $total = $fees = $net = 0;
                $agentName = $agentMap[$agentId] ?? 'Unknown';
                $stmt = $conn->prepare("SELECT fee_percent FROM submerchants WHERE cycle_agent_id = ? LIMIT 1");
                $stmt->bind_param("i", $agentId);
                $stmt->execute();
                $res = $stmt->get_result();
                $feePercent = $res->fetch_assoc()['fee_percent'] ?? 2.5;

                foreach ($txs as $tx) {
                    if (($tx['Substate'] ?? null) == 411) {
                        $amt = $tx['Amount'];
                        $fee = $amt * ($feePercent / 100);
                        $total += $amt;
                        $fees += $fee;
                        $net += $amt - $fee;
                    }
                }

                $settled = false;
                $check = $conn->query("SELECT * FROM settlements WHERE submerchant_id = $agentId AND period_start = '$date' LIMIT 1");
                if ($check && $check->num_rows > 0) {
                    $settled = $check->fetch_assoc()['payout_status'] === 'Paid';
                }

                $percent = $total > 0 ? round(($net / $total) * 100) : 0;
            ?>

            <div class="bg-white rounded shadow mb-6 overflow-hidden">
                <div class="flex justify-between items-center px-4 py-3 bg-gray-100">
                    <div class="font-semibold"> <?= $agentName ?> - <?= $date ?> </div>
                    <div class="text-sm">
                        Volume: <?= formatAmount($total) ?> | Fees: <?= formatAmount($fees) ?> | Net: <?= formatAmount($net) ?>
                    </div>
                    <?php if (!$settled): ?>
                        <form method="POST" action="mark_settled.php">
                            <input type="hidden" name="agent_id" value="<?= $agentId ?>">
                            <input type="hidden" name="date" value="<?= $date ?>">
                            <button type="submit" class="text-blue-600 hover:underline ml-4">Mark as Settled</button>
                        </form>
                    <?php else: ?>
                        <span class="text-green-600 font-medium ml-4">✅ Settled</span>
                    <?php endif; ?>
                </div>

                <div class="w-full bg-gray-200 h-2 mt-1">
                    <div class="h-2 bg-green-500" style="width: <?= $percent ?>%"></div>
                </div>

                <table class="w-full text-sm">
                    <thead>
                        <tr class="border-b bg-gray-50">
                            <th class="p-2 text-left">Time</th>
                            <th class="p-2 text-left">Amount</th>
                            <th class="p-2 text-left">Card</th>
                            <th class="p-2 text-left">Transaction ID</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($txs as $tx): ?>
                            <tr class="border-t">
                                <td class="p-2"><?= date('H:i', strtotime($tx['Date'])) ?></td>
                                <td class="p-2"><?= formatAmount($tx['Amount']) ?></td>
                                <td class="p-2"><?= $tx['Card']['PANMasked'] ?? 'Cash' ?></td>
                                <td class="p-2 text-xs"><?= $tx['ID'] ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endforeach; ?>
    <?php endforeach; ?>
</body>
</html>
